package org.msh.utils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

/**
 * Class with utilities to help working with JSON files, data and serialization
 * <p>
 * Created by rmemoria on 2/9/15.
 */
public class JsonUtils {

    private static final Log log = LogFactory.getLog(JsonUtils.class);

    /**
     * Avoid instantiation of this class
     */
    private JsonUtils() {
        super();
    }

    public static <T> T parse(InputStream in, Class<T> type) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            return mapper.readValue(in, type);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
            throw new JsonParserException(e);
        }
    }

    public static <T> T parse(InputStream in, JavaType type) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            return mapper.readValue(in, type);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
            throw new JsonParserException(e);
        }
    }

    /**
     * Convert a json String to a given Java type
     *
     * @param jsonString the json string
     * @param type       the type to parse the file into
     * @param <T>
     * @return instance of the type T
     */
    public static <T> T parseString(String jsonString, Class<T> type) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            return mapper.readValue(jsonString, type);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new JsonParserException(e);
        }

    }

    /**
     * Convert an object to JSON string representation
     *
     * @param obj object to serialize
     * @return JSON in string format
     */
    public static String objectToJSONString(Object obj, boolean pretty) {
        ObjectMapper mapper = new ObjectMapper();

        if (pretty) {
            mapper.configure(SerializationFeature.INDENT_OUTPUT, true);
        }

        try {
            return mapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            log.error(e.getMessage(), e);
            throw new JsonParserException(e);
        }
    }

    /**
     * Convert an object to JSON string representation
     *
     * @param map object to serialize
     * @return JSON in string format
     */
    public static String mapToJSONString(Map<String, Object> map, boolean pretty) {
        ObjectMapper mapper = new ObjectMapper();

        if (pretty) {
            mapper.configure(SerializationFeature.INDENT_OUTPUT, true);
        }

        try {
            return mapper.writeValueAsString(map);
        } catch (JsonProcessingException e) {
            log.error(e.getMessage(), e);
            throw new JsonParserException(e);
        }
    }

    /**
     * Convert an JSON to HashMap<String, Object>
     */
    public static Map<String, Object> parseString(String jsonString) {
        try {
            ObjectMapper mapper = new ObjectMapper();
            TypeReference<HashMap<String,Object>> typeRef = new TypeReference<HashMap<String,Object>>() {};
            return mapper.readValue(jsonString, typeRef);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new JsonParserException(e);
        }
    }
}
